package slimeknights.tconstruct.smeltery.tileentity;

import com.google.common.collect.Lists;

import net.minecraft.block.state.IBlockState;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.Packet;
import net.minecraft.network.play.server.S35PacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.BlockPos;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ITickable;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.IFluidHandler;
import net.minecraftforge.fluids.IFluidTank;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import org.apache.logging.log4j.Logger;

import java.util.List;

import slimeknights.mantle.common.IInventoryGui;
import slimeknights.mantle.multiblock.IMasterLogic;
import slimeknights.mantle.multiblock.IServantLogic;
import slimeknights.tconstruct.common.TinkerNetwork;
import slimeknights.tconstruct.library.TinkerRegistry;
import slimeknights.tconstruct.library.Util;
import slimeknights.tconstruct.library.materials.Material;
import slimeknights.tconstruct.library.smeltery.AlloyRecipe;
import slimeknights.tconstruct.library.smeltery.ISmelteryTankHandler;
import slimeknights.tconstruct.library.smeltery.MeltingRecipe;
import slimeknights.tconstruct.library.smeltery.SmelteryTank;
import slimeknights.tconstruct.library.utils.TagUtil;
import slimeknights.tconstruct.shared.TinkerFluids;
import slimeknights.tconstruct.smeltery.TinkerSmeltery;
import slimeknights.tconstruct.smeltery.block.BlockSmelteryController;
import slimeknights.tconstruct.smeltery.client.GuiSmeltery;
import slimeknights.tconstruct.smeltery.events.TinkerSmelteryEvent;
import slimeknights.tconstruct.smeltery.inventory.ContainerSmeltery;
import slimeknights.tconstruct.smeltery.multiblock.MultiblockDetection;
import slimeknights.tconstruct.smeltery.multiblock.MultiblockSmeltery;
import slimeknights.tconstruct.smeltery.network.SmelteryFluidUpdatePacket;
import slimeknights.tconstruct.smeltery.network.SmelteryFuelUpdatePacket;
import slimeknights.tconstruct.smeltery.network.SmelteryInventoryUpdatePacket;

public class TileSmeltery extends TileHeatingStructure implements IMasterLogic, ITickable, IInventoryGui,
                                                                  ISmelteryTankHandler {

  public static final DamageSource smelteryDamage = new DamageSource("smeltery").func_76361_j();

  static final Logger log = Util.getLogger("Smeltery");

  protected static final int MAX_SIZE = 9; // 9 to allow 8x8 smelteries which hold 1 stack and 9x9 for nugget/ingot processing.
  protected static final int CAPACITY_PER_BLOCK = Material.VALUE_Ingot * 8;
  protected static final int ALLOYING_PER_TICK = 10; // how much liquid can be created per tick to make alloys

  // Info about the smeltery structure/multiblock
  public boolean active;
  public MultiblockDetection.MultiblockStructure info;
  public List<BlockPos> tanks;
  public BlockPos currentTank;
  public FluidStack currentFuel; // the fuel that was last consumed

  public BlockPos minPos; // smallest coordinate INSIDE the smeltery
  public BlockPos maxPos; // biggest coordinate INSIDE the smeltery

  // Info about the state of the smeltery. Liquids etc.
  protected SmelteryTank liquids;

  protected MultiblockSmeltery multiblock;
  protected int tick;

  public TileSmeltery() {
    super("gui.smeltery.name", 0, 1);
    multiblock = new MultiblockSmeltery(this);
    liquids = new SmelteryTank(this);
    tanks = Lists.newLinkedList();
  }

  @Override
  public void func_73660_a() {
    if(this.field_145850_b.field_72995_K) {
      return;
    }

    // are we fully formed?
    if(!isActive()) {
      // check for smeltery once per second
      if(tick == 0) {
        checkSmelteryStructure();
      }
    }
    else {
      // smeltery structure is there.. do stuff with the current fuel
      // this also updates the needsFuel flag, which causes us to consume fuel at the end.
      // This way fuel is only consumed if it's actually needed

      if(tick == 0) {
        interactWithEntitiesInside();
      }
      if(tick % 4 == 0) {
        heatItems();
        alloyAlloys();
      }

      if(needsFuel) {
        consumeFuel();
      }
    }

    tick = (tick + 1) % 20;
  }

  /* Smeltery processing logic. Consuming fuel, heating stuff, creating alloys etc. */

  protected void updateHeatRequired(int index) {
    ItemStack stack = func_70301_a(index);
    if(stack != null) {
      MeltingRecipe melting = TinkerRegistry.getMelting(stack);
      if(melting != null) {
        setHeatRequiredForSlot(index, Math.max(5, melting.getUsableTemperature()));

        // instantly consume fuel if required
        if(!hasFuel()) {
          consumeFuel();
        }

        return;
      }
    }

    setHeatRequiredForSlot(index, 0);
  }

  // melt stuff
  @Override
  protected boolean onItemFinishedHeating(ItemStack stack, int slot) {
    MeltingRecipe recipe = TinkerRegistry.getMelting(stack);

    if(recipe == null) return false;

    TinkerSmelteryEvent.OnMelting event = TinkerSmelteryEvent.OnMelting.fireEvent(this, stack, recipe.output);

    int filled = liquids.fill(event.result, false);

    if(filled == event.result.amount) {
      liquids.fill(event.result, true);

      // only clear out items n stuff if it was successful
      func_70299_a(slot, null);
      return true;
    }
    else {
      // can't fill into the smeltery, set error state
      itemTemperatures[slot] = itemTempRequired[slot] * 2 + 1;
    }

    return false;
  }

  // This is how you get blisters
  protected void interactWithEntitiesInside() {
    // find all entities inside the smeltery

    AxisAlignedBB bb = info.getBoundingBox().func_72331_e(1, 1, 1);

    List<Entity> entities = field_145850_b.func_72872_a(Entity.class, bb);
    for(Entity entity : entities) {
      // item?
      if(entity instanceof EntityItem) {
        if(TinkerRegistry.getMelting(((EntityItem) entity).func_92059_d()) != null) {
          ItemStack stack = ((EntityItem) entity).func_92059_d();
          // pick it up if we can melt it
          for(int i = 0; i < this.func_70302_i_(); i++) {
            if(!isStackInSlot(i)) {
              // remove 1 from the stack and add it to the smeltery
              ItemStack invStack = stack.func_77946_l();
              stack.field_77994_a--;
              invStack.field_77994_a = 1;
              this.func_70299_a(i, invStack);
            }
            if(stack.field_77994_a <= 0) {
              // picked up whole stack
              entity.func_70106_y();
              break;
            }
          }
        }
      }
      else {
        // custom melting?
        FluidStack fluid = TinkerRegistry.getMeltingForEntity(entity);
        // no custom melting but a living entity that's alive?
        if(fluid == null && entity instanceof EntityLivingBase) {
          if(entity.func_70089_S() && !entity.field_70128_L) {
            fluid = new FluidStack(TinkerFluids.blood, 10);
          }
        }

        if(fluid != null) {
          // hurt it
          if(entity.func_70097_a(smelteryDamage, 2f)) {
            // spill the blood
            liquids.fill(fluid, true);
          }
        }
      }
    }
  }

  // check for alloys and create them
  protected void alloyAlloys() {
    for(AlloyRecipe recipe : TinkerRegistry.getAlloys()) {
        // find out how often we can apply the recipe
        int matched = recipe.matches(liquids.getFluids());
        if(matched > ALLOYING_PER_TICK) {
          matched = ALLOYING_PER_TICK;
        }
        while(matched > 0) {
          // remove all liquids from the tank
          for(FluidStack liquid : recipe.getFluids()) {
            FluidStack toDrain = liquid.copy();
            FluidStack drained = liquids.drain(toDrain, true);
            // error logging
            if(!drained.isFluidEqual(toDrain) || drained.amount != toDrain.amount) {
              log.error("Smeltery alloy creation drained incorrect amount: was %s:%d, should be %s:%d", drained
                  .getUnlocalizedName(), drained.amount, toDrain.getUnlocalizedName(), toDrain.amount);
            }
          }

          // and insert the alloy
          FluidStack toFill = recipe.getResult().copy();
          int filled = liquids.fill(toFill, true);
          if(filled != recipe.getResult().amount) {
            log.error("Smeltery alloy creation filled incorrect amount: was %d, should be %d (%s)", filled,
                      recipe.getResult().amount * matched, recipe.getResult().getUnlocalizedName());
          }
          matched -= filled;
      }
    }
  }

  @Override
  protected void consumeFuel() {
    // no need to consume fuel
    if(hasFuel()) {
      return;
    }

    // get current tank
    searchForFuel();

    // got a tank?
    if(currentTank != null) {
      // consume fuel!
      TileEntity te = field_145850_b.func_175625_s(currentTank);
      if(te instanceof TileTank) {
        TileTank tank = (TileTank) te;

        FluidStack liquid = tank.getInternalTank().getFluid();
        if(liquid != null) {
          FluidStack in = liquid.copy();
          int bonusFuel = TinkerRegistry.consumeSmelteryFuel(in);
          int amount = liquid.amount - in.amount;
          FluidStack drained = tank.drain(null, amount, false);

          // we can drain. actually drain and add the fuel
          if(drained.amount == amount) {
            tank.drain(null, amount, true);
            currentFuel = drained.copy();
            addFuel(bonusFuel, drained.getFluid().getTemperature(drained) - 300); // convert to degree celcius

            // notify client of fuel/temperature changes
            if(field_145850_b != null && !field_145850_b.field_72995_K) {
              TinkerNetwork.sendToAll(new SmelteryFuelUpdatePacket(field_174879_c, currentTank, temperature, currentFuel));
            }
          }
        }
      }
    }
  }

  private void searchForFuel() {
    // is the current tank still up to date?
    if(currentTank != null && hasFuel(currentTank, currentFuel)) {
      return;
    }

    // nope, current tank is empty, check others for same fuel
    for(BlockPos pos : tanks) {
      if(hasFuel(pos, currentFuel)) {
        currentTank = pos;
        return;
      }
    }

    // nothing found, try again with new fuel
    for(BlockPos pos : tanks) {
      if(hasFuel(pos, null)) {
        currentTank = pos;
        return;
      }
    }

    currentTank = null;
  }

  // checks if the given location has a fluid tank that contains fuel
  private boolean hasFuel(BlockPos pos, FluidStack preference) {
    IFluidTank tank = getTankAt(pos);
    if(tank != null) {
      if(tank.getFluidAmount() > 0 && TinkerRegistry.isSmelteryFuel(tank.getFluid())) {
        // if we have a preference, only use that
        if(preference != null && tank.getFluid().isFluidEqual(preference)) {
          return true;
        }
        else if(preference == null) {
          return true;
        }
      }
    }

    return false;
  }

  private IFluidTank getTankAt(BlockPos pos) {
    TileEntity te = field_145850_b.func_175625_s(pos);
    if(te instanceof TileTank) {
      return ((TileTank) te).getInternalTank();
    }

    return null;
  }

  /* Smeltery Multiblock Detection/Formation */

  /** Called by the servants */
  @Override
  public void notifyChange(IServantLogic servant, BlockPos pos) {
    checkSmelteryStructure();
  }

  // Checks if the smeltery is fully built and updates status accordingly
  public void checkSmelteryStructure() {
    boolean wasActive = isActive();

    IBlockState state = this.field_145850_b.func_180495_p(func_174877_v());
    if(!(state.func_177230_c() instanceof BlockSmelteryController)) {
      active = false;
    }
    else {
      EnumFacing in = state.func_177229_b(BlockSmelteryController.FACING).func_176734_d();

      MultiblockDetection.MultiblockStructure structure = multiblock.detectMultiblock(this.field_145850_b, this.func_174877_v().func_177972_a(in), MAX_SIZE);
      if(structure == null) {
        active = false;
        updateSmelteryInfo(new MultiblockDetection.MultiblockStructure(0, 0, 0, Lists.<BlockPos>newLinkedList()));
      }
      else {
        // we found a valid smeltery. yay.
        active = true;
        MultiblockDetection.assignMultiBlock(this.field_145850_b, this.func_174877_v(), structure.blocks);
        updateSmelteryInfo(structure);
        // we still have to update since something caused us to rebuild our stats
        // might be the smeltery size changed
        if(wasActive)
          field_145850_b.func_175689_h(field_174879_c);
      }
    }

    // mark the block for updating so the smeltery controller block updates its graphics
    if(wasActive != isActive()) {
      field_145850_b.func_175689_h(field_174879_c);
      this.func_70296_d();
    }
  }

  protected void updateSmelteryInfo(MultiblockDetection.MultiblockStructure structure) {
    info = structure;

    if(info != null) {
      minPos = info.minPos.func_177982_a(1,1,1); // add walls and floor
      maxPos = info.maxPos.func_177982_a(-1, 0, -1); // subtract walls, no ceiling
    }
    else {
      minPos = maxPos = this.field_174879_c;
    }

    // find all tanks for input
    tanks.clear();
    for(BlockPos pos : structure.blocks) {
      if(field_145850_b.func_180495_p(pos).func_177230_c() == TinkerSmeltery.searedTank) {
        tanks.add(pos);
      }
    }

    int inventorySize = structure.xd * structure.yd * structure.zd;
    // if the new smeltery is smaller we pop out all items that don't fit in anymore
    if(this.func_70302_i_() > inventorySize) {
      for(int i = inventorySize; i < func_70302_i_(); i++) {
        if(func_70301_a(i) != null) {
          dropItem(func_70301_a(i));
        }
      }
    }

    this.liquids.setCapacity(inventorySize * CAPACITY_PER_BLOCK);

    // adjust inventory sizes
    this.resize(inventorySize);

    //System.out.println(String.format("[%s] Smeltery detected. Size: %d x %d x %d, %d slots", worldObj != null && worldObj.isRemote ? "Client" : "Server", structure.xd, structure.zd, structure.yd, inventorySize));
  }

  private void dropItem(ItemStack stack) {
    EnumFacing direction = field_145850_b.func_180495_p(field_174879_c).func_177229_b(BlockSmelteryController.FACING);
    BlockPos pos = this.func_174877_v().func_177972_a(direction);

    EntityItem entityitem = new EntityItem(field_145850_b, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), stack);
    field_145850_b.func_72838_d(entityitem);
  }



  /* Fluid handling */

  public SmelteryTank getTank() {
    return liquids;
  }

  /* GUI */
  @Override
  public Container createContainer(InventoryPlayer inventoryplayer, World world, BlockPos pos) {
    return new ContainerSmeltery(inventoryplayer, this);
  }

  @Override
  @SideOnly(Side.CLIENT)
  public GuiContainer createGui(InventoryPlayer inventoryplayer, World world, BlockPos pos) {
    return new GuiSmeltery((ContainerSmeltery)createContainer(inventoryplayer, world, pos), this);
  }

  public float getMeltingProgress(int index) {
    if(index < 0 || index > func_70302_i_() - 1) {
      return -1f;
    }

    if(!canHeat(index)) {
      return -1f;
    }

    return getProgress(index);
  }

  @SideOnly(Side.CLIENT)
  public FuelInfo getFuelDisplay() {
    FuelInfo info = new FuelInfo();

    // we still have leftover fuel
    if(hasFuel()) {
      info.fluid = currentFuel.copy();
      info.fluid.amount = 0;
      info.heat = this.temperature;
      info.maxCap = currentFuel.amount;
    }
    else if(currentTank != null) {
      // we need to consume fuel, check the current tank
      if(hasFuel(currentTank, currentFuel)) {
        IFluidTank tank = getTankAt(currentTank);
        info.fluid = tank.getFluid().copy();
        info.heat = temperature;
        info.maxCap = tank.getCapacity();
      }
    }

    // check all other tanks (except the current one that we already checked) for more fuel
    for(BlockPos pos : tanks) {
      if(pos == currentTank) continue;

      IFluidTank tank = getTankAt(pos);
      // tank exists and has something in it
      if(tank != null && tank.getFluidAmount() > 0) {
        // we don't have fuel yet, use this
        if(info.fluid == null) {
          info.fluid = tank.getFluid().copy();
          info.heat = info.fluid.getFluid().getTemperature(info.fluid);
          info.maxCap = tank.getCapacity();
        }
        // otherwise add the same together
        else if(tank.getFluid().isFluidEqual(info.fluid)) {
          info.fluid.amount += tank.getFluidAmount();
          info.maxCap += tank.getCapacity();
        }
      }
    }

    return info;
  }

  @Override
  public AxisAlignedBB getRenderBoundingBox() {
    if(minPos == null || maxPos == null) {
      return super.getRenderBoundingBox();
    }
    return AxisAlignedBB.func_178781_a(minPos.func_177958_n(), minPos.func_177956_o(), minPos.func_177952_p(), maxPos.func_177958_n()+1, maxPos.func_177956_o()+1, maxPos.func_177952_p()+1);
  }

  /* Network & Saving */

  @Override
  public void func_70299_a(int slot, ItemStack itemstack) {
    // send to client if needed
    if(this.field_145850_b != null && this.field_145850_b instanceof WorldServer && !this.field_145850_b.field_72995_K && !ItemStack.func_77989_b(itemstack, func_70301_a(slot))) {
      TinkerNetwork.sendToClients((WorldServer) this.field_145850_b, this.field_174879_c, new SmelteryInventoryUpdatePacket(itemstack, slot, field_174879_c));
    }
    super.func_70299_a(slot, itemstack);
  }

  @SideOnly(Side.CLIENT)
  public void updateTemperatureFromPacket(int index, int heat) {
    if(index < 0 || index > func_70302_i_()-1) {
      return;
    }

    itemTemperatures[index] = heat;
  }

  @SideOnly(Side.CLIENT)
  public void updateFluidsFromPacket(List<FluidStack> fluids) {
    this.liquids.setFluids(fluids);
  }

  @Override
  public void onTankChanged(List<FluidStack> fluids, FluidStack changed) {
    // notify clients of liquid changes.
    // the null check is to prevent potential crashes during loading
    if(field_145850_b != null && !field_145850_b.field_72995_K) {
      TinkerNetwork.sendToAll(new SmelteryFluidUpdatePacket(field_174879_c, fluids));
    }
  }

  @Override
  public void func_145829_t() {
    super.func_145829_t();
    // on validation we set active to false so the smeltery checks anew if it's formed
    active = false;
  }

  @Override
  public void func_145841_b(NBTTagCompound compound) {
    super.func_145841_b(compound);
    liquids.writeToNBT(compound);

    compound.func_74757_a("active", active);
    compound.func_74782_a("currentTank", TagUtil.writePos(currentTank));
    NBTTagList tankList = new NBTTagList();
    for(BlockPos pos : tanks) {
      tankList.func_74742_a(TagUtil.writePos(pos));
    }
    compound.func_74782_a("tanks", tankList);

    NBTTagCompound fuelTag = new NBTTagCompound();
    if(currentFuel != null) {
      currentFuel.writeToNBT(fuelTag);
    }
    compound.func_74782_a("currentFuel", fuelTag);

    compound.func_74782_a("minPos", TagUtil.writePos(minPos));
    compound.func_74782_a("maxPos", TagUtil.writePos(maxPos));
  }

  @Override
  public void func_145839_a(NBTTagCompound compound) {
    super.func_145839_a(compound);
    liquids.readFromNBT(compound);

    active = compound.func_74767_n("active");
    NBTTagList tankList = compound.func_150295_c("tanks", 10);
    tanks.clear();
    for(int i = 0; i < tankList.func_74745_c(); i++) {
      tanks.add(TagUtil.readPos(tankList.func_150305_b(i)));
    }

    NBTTagCompound fuelTag = compound.func_74775_l("currentFuel");
    currentFuel = FluidStack.loadFluidStackFromNBT(fuelTag);

    minPos = TagUtil.readPos(compound.func_74775_l("minPos"));
    maxPos = TagUtil.readPos(compound.func_74775_l("maxPos"));
  }

  @Override
  public Packet func_145844_m() {
    NBTTagCompound tag = new NBTTagCompound();
    func_145841_b(tag);
    return new S35PacketUpdateTileEntity(this.func_174877_v(), this.func_145832_p(), tag);
  }

  @Override
  public void onDataPacket(NetworkManager net, S35PacketUpdateTileEntity pkt) {
    boolean wasActive = active;

    func_145839_a(pkt.func_148857_g());

    // update chunk (rendering) if the active state changed
    if(isActive() != wasActive) {
      field_145850_b.func_175689_h(field_174879_c);
    }
  }

  /* Getter */

  public boolean isActive() {
    return active;
  }

  public static class FuelInfo {
    public int heat;
    public int maxCap;
    public FluidStack fluid;
  }
}
