package slimeknights.tconstruct.smeltery;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;

import net.minecraft.block.Block;
import net.minecraft.entity.monster.EntityIronGolem;
import net.minecraft.entity.monster.EntitySnowman;
import net.minecraft.entity.passive.EntityVillager;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.CraftingManager;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.item.crafting.ShapedRecipes;
import net.minecraft.item.crafting.ShapelessRecipes;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidContainerRegistry;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fml.common.SidedProxy;
import net.minecraftforge.fml.common.event.FMLInitializationEvent;
import net.minecraftforge.fml.common.event.FMLPostInitializationEvent;
import net.minecraftforge.fml.common.event.FMLPreInitializationEvent;
import net.minecraftforge.fml.common.registry.GameRegistry;
import net.minecraftforge.oredict.OreDictionary;
import net.minecraftforge.oredict.ShapedOreRecipe;
import net.minecraftforge.oredict.ShapelessOreRecipe;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.Logger;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import slimeknights.mantle.item.ItemBlockMeta;
import slimeknights.mantle.pulsar.pulse.Pulse;
import slimeknights.mantle.util.RecipeMatch;
import slimeknights.tconstruct.common.CommonProxy;
import slimeknights.tconstruct.common.TinkerPulse;
import slimeknights.tconstruct.common.config.Config;
import slimeknights.tconstruct.library.TinkerRegistry;
import slimeknights.tconstruct.library.Util;
import slimeknights.tconstruct.library.materials.Material;
import slimeknights.tconstruct.library.smeltery.Cast;
import slimeknights.tconstruct.library.smeltery.CastingRecipe;
import slimeknights.tconstruct.library.smeltery.MeltingRecipe;
import slimeknights.tconstruct.library.smeltery.OreCastingRecipe;
import slimeknights.tconstruct.library.tinkering.MaterialItem;
import slimeknights.tconstruct.library.tools.IToolPart;
import slimeknights.tconstruct.shared.TinkerCommons;
import slimeknights.tconstruct.shared.TinkerFluids;
import slimeknights.tconstruct.smeltery.block.BlockCasting;
import slimeknights.tconstruct.smeltery.block.BlockFaucet;
import slimeknights.tconstruct.smeltery.block.BlockSeared;
import slimeknights.tconstruct.smeltery.block.BlockSmelteryController;
import slimeknights.tconstruct.smeltery.block.BlockSmelteryIO;
import slimeknights.tconstruct.smeltery.block.BlockTank;
import slimeknights.tconstruct.smeltery.item.CastCustom;
import slimeknights.tconstruct.smeltery.item.ItemTank;
import slimeknights.tconstruct.smeltery.tileentity.TileCastingBasin;
import slimeknights.tconstruct.smeltery.tileentity.TileCastingTable;
import slimeknights.tconstruct.smeltery.tileentity.TileDrain;
import slimeknights.tconstruct.smeltery.tileentity.TileFaucet;
import slimeknights.tconstruct.smeltery.tileentity.TileSmeltery;
import slimeknights.tconstruct.smeltery.tileentity.TileSmelteryComponent;
import slimeknights.tconstruct.smeltery.tileentity.TileTank;
import slimeknights.tconstruct.tools.TinkerMaterials;
import slimeknights.tconstruct.world.TinkerWorld;
import slimeknights.tconstruct.world.block.BlockSlime;

@Pulse(id = TinkerSmeltery.PulseId, description = "The smeltery and items needed for it")
public class TinkerSmeltery extends TinkerPulse {

  public static final String PulseId = "TinkerSmeltery";
  static final Logger log = Util.getLogger(PulseId);

  @SidedProxy(clientSide = "slimeknights.tconstruct.smeltery.SmelteryClientProxy", serverSide = "slimeknights.tconstruct.common.CommonProxy")
  public static CommonProxy proxy;

  // Blocks
  public static BlockSeared searedBlock;
  public static BlockSmelteryController smelteryController;
  public static BlockTank searedTank;
  public static BlockFaucet faucet;
  public static BlockCasting castingBlock;
  public static BlockSmelteryIO smelteryIO;

  // Items
  public static Cast cast;
  public static CastCustom castCustom;
  public static Cast clayCast;

  // itemstacks!
  public static ItemStack castIngot;
  public static ItemStack castNugget;
  public static ItemStack castGem;
  public static ItemStack castShard;
  public static ItemStack castPlate;
  public static ItemStack castGear;

  private static Map<Fluid, Set<Pair<List<ItemStack>, Integer>>> knownOreFluids = Maps.newHashMap();
  public static List<FluidStack> castCreationFluids = Lists.newLinkedList();
  public static List<FluidStack> clayCreationFluids = Lists.newLinkedList();

  public static ImmutableSet<Block> validSmelteryBlocks;
  public static List<ItemStack> meltingBlacklist = Lists.newLinkedList();

  // PRE-INITIALIZATION
  @Subscribe
  public void preInit(FMLPreInitializationEvent event) {
    searedBlock = registerEnumBlock(new BlockSeared(), "seared");
    smelteryController = registerBlock(new BlockSmelteryController(), "smeltery_controller");
    searedTank = registerBlock(new BlockTank(), ItemTank.class, "seared_tank");
    faucet = registerBlock(new BlockFaucet(), "faucet");
    castingBlock = registerBlock(new BlockCasting(), ItemBlockMeta.class, "casting");
    smelteryIO = registerEnumBlock(new BlockSmelteryIO(), "smeltery_io");

    ItemBlockMeta.setMappingProperty(searedTank, BlockTank.TYPE);
    ItemBlockMeta.setMappingProperty(castingBlock, BlockCasting.TYPE);

    registerTE(TileSmeltery.class, "smeltery_controller");
    registerTE(TileSmelteryComponent.class, "smeltery_component");
    registerTE(TileTank.class, "tank");
    registerTE(TileFaucet.class, "faucet");
    registerTE(TileCastingTable.class, "casting_table");
    registerTE(TileCastingBasin.class, "casting_basin");
    registerTE(TileDrain.class, "smeltery_drain");

    cast = registerItem(new Cast(), "cast");
    castCustom = registerItem(new CastCustom(), "cast_custom");
    castIngot = castCustom.addMeta(0, "ingot", Material.VALUE_Ingot);
    castNugget = castCustom.addMeta(1, "nugget", Material.VALUE_Nugget);
    castGem = castCustom.addMeta(2, "gem", Material.VALUE_Gem);
    castPlate = castCustom.addMeta(3, "plate", Material.VALUE_Ingot);
    castGear = castCustom.addMeta(4, "gear", Material.VALUE_Ingot*4);

    clayCast = registerItem(new Cast(), "clay_cast");

    if(TinkerRegistry.getShard() != null) {
      TinkerRegistry.addCastForItem(TinkerRegistry.getShard());
      castShard = new ItemStack(cast);
      Cast.setTagForPart(castShard, TinkerRegistry.getShard());
    }

    proxy.preInit();

    TinkerRegistry.tabSmeltery.setDisplayIcon(new ItemStack(searedTank));

    ImmutableSet.Builder<Block> builder = ImmutableSet.builder();
    builder.add(searedBlock);
    builder.add(searedTank);
    builder.add(smelteryIO);

    validSmelteryBlocks = builder.build();
  }

  // INITIALIZATION
  @Subscribe
  public void init(FMLInitializationEvent event) {
    // done here so they're present for integration in MaterialIntegration and fluids in TinkerFluids are also initialized
    castCreationFluids.add(new FluidStack(TinkerFluids.gold, Material.VALUE_Ingot*2));
    if(FluidRegistry.isFluidRegistered(TinkerFluids.brass)) {
      castCreationFluids.add(new FluidStack(TinkerFluids.brass, Material.VALUE_Ingot));
    }
    if(FluidRegistry.isFluidRegistered(TinkerFluids.clay)) {
      clayCreationFluids.add(new FluidStack(TinkerFluids.clay, Material.VALUE_Ingot*2));
    }

    registerRecipes();

    proxy.init();
  }

  private void registerRecipes() {
    // I AM GROUT
    ItemStack grout = TinkerCommons.grout.func_77946_l();
    grout.field_77994_a = 2;
    GameRegistry.addRecipe(new ShapelessOreRecipe(grout, Items.field_151119_aD, Blocks.field_150351_n, "sand"));
    grout = grout.func_77946_l();
    grout.field_77994_a = 8;
    GameRegistry.addRecipe(new ShapelessOreRecipe(grout, Blocks.field_150351_n, "sand", Blocks.field_150351_n, "sand",  Blocks.field_150435_aG, "sand", Blocks.field_150351_n, "sand", Blocks.field_150351_n));

    // seared bricks
    GameRegistry.addSmelting(TinkerCommons.grout, TinkerCommons.searedBrick, 0);
    ItemStack blockSeared = new ItemStack(searedBlock);
    blockSeared.func_77964_b(BlockSeared.SearedType.BRICK.getMeta());
    GameRegistry.addShapedRecipe(blockSeared, "bb", "bb", 'b', TinkerCommons.searedBrick);

    // remaining smeltery component recipes
    ItemStack searedBrick = TinkerCommons.searedBrick;
    GameRegistry.addRecipe(new ItemStack(smelteryController),
                           "bbb", "b b", "bbb", 'b', searedBrick); // Controller
    GameRegistry.addRecipe(new ItemStack(smelteryIO, 1, BlockSmelteryIO.IOType.DRAIN.getMeta()),
                           "b b", "b b", "b b", 'b', searedBrick); // Drain
    GameRegistry.addRecipe(new ShapedOreRecipe(new ItemStack(searedTank, 1, BlockTank.TankType.TANK.getMeta()),
                                               "bbb", "bgb", "bbb", 'b', searedBrick, 'g', "blockGlass")); // Tank
    GameRegistry.addRecipe(new ShapedOreRecipe(new ItemStack(searedTank, 1, BlockTank.TankType.GAUGE.getMeta()),
                                               "bgb", "ggg", "bgb", 'b', searedBrick, 'g', "blockGlass")); // Glass
    GameRegistry.addRecipe(new ShapedOreRecipe(new ItemStack(searedTank, 1, BlockTank.TankType.WINDOW.getMeta()),
                                               "bgb", "bgb", "bgb", 'b', searedBrick, 'g', "blockGlass")); // Window

    GameRegistry.addRecipe(new ItemStack(castingBlock, 1, BlockCasting.CastingType.TABLE.getMeta()),
                           "bbb", "b b", "b b", 'b', searedBrick); // Table
    GameRegistry.addRecipe(new ItemStack(castingBlock, 1, BlockCasting.CastingType.BASIN.getMeta()),
                           "b b", "b b", "bbb", 'b', searedBrick); // Basin
    GameRegistry.addRecipe(new ItemStack(faucet),
                           "b b", " b ", 'b', searedBrick); // Faucet
    //GameRegistry.addRecipe(new ItemStack(TinkerSmeltery.castingChannel, 4, 0), "b b", "bbb", 'b', searedBrick); // Channel
  }

  // POST-INITIALIZATION
  @Subscribe
  public void postInit(FMLPostInitializationEvent event) {
    registerSmelteryFuel();
    registerMeltingCasting();
    registerAlloys();

    registerRecipeOredictMelting();

    // register remaining cast creation
    for(FluidStack fs : castCreationFluids) {
      TinkerRegistry.registerTableCasting(new ItemStack(cast), null, fs.getFluid(), fs.amount);
      TinkerRegistry.registerTableCasting(new CastingRecipe(castGem, RecipeMatch.of("gemEmerald"), fs, true, true));
      TinkerRegistry.registerTableCasting(new CastingRecipe(castIngot, RecipeMatch.of("ingotBrick"), fs, true, true));
      TinkerRegistry.registerTableCasting(new CastingRecipe(castIngot, RecipeMatch.of("ingotBrickNether"), fs, true, true));
      TinkerRegistry.registerTableCasting(new CastingRecipe(castIngot, new RecipeMatch.Item(TinkerCommons.searedBrick, 1), fs, true, true));
    }

    proxy.postInit();
  }

  private void registerSmelteryFuel() {
    TinkerRegistry.registerSmelteryFuel(new FluidStack(FluidRegistry.LAVA, 50), 100);
  }

  private void registerMeltingCasting() {
    int bucket = FluidContainerRegistry.BUCKET_VOLUME;

    // Water
    Fluid water = FluidRegistry.WATER;
    TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of(Blocks.field_150432_aD, bucket), water, 305));
    TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of(Blocks.field_150403_cj, bucket * 2), water, 310));
    TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of(Blocks.field_150433_aE, bucket), water, 305));
    TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of(Items.field_151126_ay, bucket / 8), water, 301));

    // bloooooood
    TinkerRegistry.registerMelting(Items.field_151078_bh, TinkerFluids.blood, 5);
    if(TinkerCommons.matSlimeBallBlood != null) {
      TinkerRegistry.registerTableCasting(TinkerCommons.matSlimeBallBlood.func_77946_l(), null, TinkerFluids.blood, 160);
    }

    // purple slime
    TinkerRegistry.registerMelting(TinkerCommons.matSlimeBallPurple, TinkerFluids.purpleSlime, Material.VALUE_SlimeBall);
    if(TinkerWorld.slimeBlockCongealed != null) {
      ItemStack slimeblock = new ItemStack(TinkerWorld.slimeBlockCongealed, 1, BlockSlime.SlimeType.PURPLE.meta);
      TinkerRegistry.registerMelting(slimeblock, TinkerFluids.purpleSlime, Material.VALUE_SlimeBall * 4);
      slimeblock = new ItemStack(TinkerWorld.slimeBlock, 1, BlockSlime.SlimeType.PURPLE.meta);
      TinkerRegistry.registerMelting(slimeblock, TinkerFluids.purpleSlime, Material.VALUE_SlimeBall * 9);
    }

    // seared stone, takes as long as a full block to melt, but gives less
    TinkerRegistry.registerMelting(MeltingRecipe.forAmount(RecipeMatch.of("stone", Material.VALUE_SearedMaterial),
                                                           TinkerFluids.searedStone, Material.VALUE_Ore));
    TinkerRegistry.registerMelting(MeltingRecipe.forAmount(RecipeMatch.of("cobblestone", Material.VALUE_SearedMaterial),
                                                           TinkerFluids.searedStone, Material.VALUE_Ore));

    // obsidian
    TinkerRegistry.registerMelting(MeltingRecipe.forAmount(RecipeMatch.of("obsidian", Material.VALUE_Ore),
                                                           TinkerFluids.obsidian, Material.VALUE_Ore));
    // note that obsidian casting gives you 2 ingot value per obsidian, while part crafting only gives 1 per obsidian
    registerToolpartMeltingCasting(TinkerMaterials.obsidian);
    TinkerRegistry.registerBasinCasting(new ItemStack(Blocks.field_150343_Z), null, TinkerFluids.obsidian, Material.VALUE_Ore);


    // gold is melt and castable too, but no tools. Remaining materials are done directly in the MaterialIntegration
    // gold is integrated via MaterialIntegration in TinkerIntegration now

    // register stone toolpart melting
    for(IToolPart toolPart : TinkerRegistry.getToolParts()) {
      if(toolPart instanceof MaterialItem) {
        ItemStack stack = toolPart.getItemstackWithMaterial(TinkerMaterials.stone);
        TinkerRegistry.registerMelting(stack, TinkerFluids.searedStone, toolPart.getCost());
      }
    }

    // seared block casting and melting
    ItemStack blockSeared = new ItemStack(searedBlock);
    blockSeared.func_77964_b(BlockSeared.SearedType.STONE.getMeta());
    TinkerRegistry.registerTableCasting(TinkerCommons.searedBrick, castIngot, TinkerFluids.searedStone, Material.VALUE_SearedMaterial);
    TinkerRegistry.registerBasinCasting(blockSeared, null, TinkerFluids.searedStone, Material.VALUE_SearedBlock);
    // basically a pseudo-oredict of the seared blocks to support wildcard value
    TinkerRegistry.registerMelting(searedBlock, TinkerFluids.searedStone, Material.VALUE_SearedBlock);
    TinkerRegistry.registerMelting(TinkerCommons.searedBrick, TinkerFluids.searedStone, Material.VALUE_SearedMaterial);

    // melt all the dirt into mud
    ItemStack stack = new ItemStack(Blocks.field_150346_d, 1, OreDictionary.WILDCARD_VALUE);
    RecipeMatch rm = new RecipeMatch.Item(stack, 1, Material.VALUE_Ingot);
    TinkerRegistry.registerMelting(MeltingRecipe.forAmount(rm, TinkerFluids.dirt, Material.VALUE_BrickBlock));
    TinkerRegistry.registerTableCasting(TinkerCommons.mudBrick, castIngot, TinkerFluids.dirt, Material.VALUE_Ingot);

    // hardened clay
    TinkerRegistry.registerMelting(Items.field_151119_aD, TinkerFluids.clay, Material.VALUE_Ingot);
    TinkerRegistry.registerMelting(Blocks.field_150435_aG, TinkerFluids.clay, Material.VALUE_BrickBlock);
    // decided against support for melting hardened clay. Once it's hardened, it stays hard. Same for bricks.
    //TinkerRegistry.registerMelting(Blocks.hardened_clay, TinkerFluids.clay, Material.VALUE_BrickBlock);
    //TinkerRegistry.registerMelting(Blocks.stained_hardened_clay, TinkerFluids.clay, Material.VALUE_BrickBlock);
    TinkerRegistry.registerBasinCasting(new ItemStack(Blocks.field_150405_ch), null, TinkerFluids.clay, Material.VALUE_BrickBlock);

    // emerald melting and casting
    TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of("gemEmerald", Material.VALUE_Gem), TinkerFluids.emerald));
    TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of("blockEmerald", Material.VALUE_Gem*9), TinkerFluids.emerald));
    TinkerRegistry.registerTableCasting(new ItemStack(Items.field_151166_bC), castGem, TinkerFluids.emerald, Material.VALUE_Gem);
    TinkerRegistry.registerBasinCasting(new ItemStack(Blocks.field_150475_bE), null, TinkerFluids.emerald, Material.VALUE_Gem*9);

    // lavawood
    TinkerRegistry.registerBasinCasting(new CastingRecipe(TinkerCommons.lavawood, RecipeMatch.of("plankWood"),
                                                          new FluidStack(FluidRegistry.LAVA, 250),
                                                          100, true, false));


    // red sand
    TinkerRegistry.registerBasinCasting(new ItemStack(Blocks.field_150354_m, 1, 1), new ItemStack(Blocks.field_150354_m, 1, 0), TinkerFluids.blood, 10);

    // melt entities into a pulp
    TinkerRegistry.registerEntityMelting(EntityIronGolem.class, new FluidStack(TinkerFluids.iron, 18));
    TinkerRegistry.registerEntityMelting(EntitySnowman.class, new FluidStack(FluidRegistry.WATER, 100));
    TinkerRegistry.registerEntityMelting(EntityVillager.class, new FluidStack(TinkerFluids.emerald, 6));
  }

  private void registerAlloys() {
    // 1 bucket lava + 1 bucket water = 2 ingots = 1 block obsidian
    // 1000 + 1000 = 288
    // 125 + 125 = 36
    if(Config.obsidianAlloy) {
      TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.obsidian, 36),
                                   new FluidStack(FluidRegistry.WATER, 125),
                                   new FluidStack(FluidRegistry.LAVA, 125));
    }

    // 1 bucket water + 4 seared ingot + 4 mud bricks = 1 block hardened clay
    // 1000 + 288 + 576 = 576
    // 250 + 72 + 144 = 144
    TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.clay, 144),
                                 new FluidStack(FluidRegistry.WATER, 250),
                                 new FluidStack(TinkerFluids.searedStone, 72),
                                 new FluidStack(TinkerFluids.dirt, 144));

    // 1 iron ingot + 1 purple slime ball + seared stone in molten form = 1 knightslime ingot
    // 144 + 250 + 288 = 144
    TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.knightslime, 72),
                                 new FluidStack(TinkerFluids.iron, 72),
                                 new FluidStack(TinkerFluids.purpleSlime, 125),
                                 new FluidStack(TinkerFluids.searedStone, 144));

    // i iron ingot + 1 blood... unit thingie + 1/3 gem = 1 pigiron
    // 144 + 99 + 222 = 144
    TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.pigIron, 144),
                                 new FluidStack(TinkerFluids.iron, 48),
                                 new FluidStack(TinkerFluids.blood, 33),
                                 new FluidStack(TinkerFluids.emerald, 74));

    // 1 ingot cobalt + 1 ingot ardite = 1 ingot manyullyn!
    // 144 + 144 = 144
    TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.manyullyn, 2),
                                 new FluidStack(TinkerFluids.cobalt, 2),
                                 new FluidStack(TinkerFluids.ardite, 2));

    // 3 ingots copper + 1 ingot tin = 4 ingots bronze
    if(FluidRegistry.isFluidRegistered(TinkerFluids.bronze) &&
       FluidRegistry.isFluidRegistered(TinkerFluids.copper) &&
       FluidRegistry.isFluidRegistered(TinkerFluids.tin)) {
      TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.bronze, 4),
                                   new FluidStack(TinkerFluids.copper, 3),
                                   new FluidStack(TinkerFluids.tin, 1));
    }

    // 1 ingot gold + 1 ingot silver = 2 ingots electrum
    if(FluidRegistry.isFluidRegistered(TinkerFluids.electrum) &&
       FluidRegistry.isFluidRegistered(TinkerFluids.gold) &&
       FluidRegistry.isFluidRegistered(TinkerFluids.silver)) {
      TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.electrum, 2),
                                   new FluidStack(TinkerFluids.gold, 1),
                                   new FluidStack(TinkerFluids.silver, 1));
    }

    if(FluidRegistry.isFluidRegistered(TinkerFluids.alubrass) &&
       FluidRegistry.isFluidRegistered(TinkerFluids.copper) &&
       FluidRegistry.isFluidRegistered(TinkerFluids.aluminum)) {
      TinkerRegistry.registerAlloy(new FluidStack(TinkerFluids.alubrass, 4),
                                   new FluidStack(TinkerFluids.copper, 1),
                                   new FluidStack(TinkerFluids.aluminum, 3));
    }
  }

  public static void registerToolpartMeltingCasting(Material material) {
    // melt ALL the toolparts n stuff. Also cast them.
    Fluid fluid = material.getFluid();
    for(IToolPart toolPart : TinkerRegistry.getToolParts()) {
      if(toolPart instanceof MaterialItem) {
        ItemStack stack = toolPart.getItemstackWithMaterial(material);
        ItemStack cast = new ItemStack(TinkerSmeltery.cast);
        Cast.setTagForPart(cast, stack.func_77973_b());

        if(fluid != null) {
          // melting
          TinkerRegistry.registerMelting(stack, fluid, toolPart.getCost());
          // casting
          TinkerRegistry.registerTableCasting(stack, cast, fluid, toolPart.getCost());
        }
        // register cast creation from the toolparts
        for(FluidStack fs : castCreationFluids) {
          TinkerRegistry.registerTableCasting(new CastingRecipe(cast,
                                                                RecipeMatch.ofNBT(stack),
                                                                fs,
                                                                true, true));
        }

        // clay casts
        if(Config.claycasts) {
          ItemStack clayCast = new ItemStack(TinkerSmeltery.clayCast);
          Cast.setTagForPart(clayCast, stack.func_77973_b());

          if(fluid != null) {
            RecipeMatch rm = RecipeMatch.ofNBT(clayCast);
            FluidStack fs = new FluidStack(fluid, toolPart.getCost());
            TinkerRegistry.registerTableCasting(new CastingRecipe(stack, rm, fs, true, false));
          }
          for(FluidStack fs : clayCreationFluids) {
            TinkerRegistry.registerTableCasting(new CastingRecipe(clayCast,
                                                                  RecipeMatch.ofNBT(stack),
                                                                  fs,
                                                                  true, true));
          }
        }
      }
    }

    // same for shard
    if(castShard != null) {
      ItemStack stack = TinkerRegistry.getShard(material);
      int cost = TinkerRegistry.getShard().getCost();

      if(fluid != null) {
        // melting
        TinkerRegistry.registerMelting(stack, fluid, cost);
        // casting
        TinkerRegistry.registerTableCasting(stack, castShard, fluid, cost);
      }
      // register cast creation from the toolparts
      for(FluidStack fs : castCreationFluids) {
        TinkerRegistry.registerTableCasting(new CastingRecipe(castShard,
                                                              RecipeMatch.ofNBT(stack),
                                                              fs,
                                                              true, true));
      }
    }
  }

  /**
   * Registers melting for all directly supported pre- and suffixes of the ore.
   * E.g. "Iron" -> "ingotIron", "blockIron", "oreIron",
   */
  public static void registerOredictMeltingCasting(Fluid fluid, String ore) {
    ImmutableSet.Builder<Pair<List<ItemStack>, Integer>> builder = ImmutableSet.builder();
    Pair<List<ItemStack>, Integer> nuggetOre = Pair.of(OreDictionary.getOres("nugget" + ore), Material.VALUE_Nugget);
    Pair<List<ItemStack>, Integer> ingotOre = Pair.of(OreDictionary.getOres("ingot" + ore), Material.VALUE_Ingot);
    Pair<List<ItemStack>, Integer> blockOre = Pair.of(OreDictionary.getOres("block" + ore), Material.VALUE_Block);
    Pair<List<ItemStack>, Integer> oreOre = Pair.of(OreDictionary.getOres("ore" + ore), Material.VALUE_Ore);
    Pair<List<ItemStack>, Integer> plateOre = Pair.of(OreDictionary.getOres("plate" + ore), Material.VALUE_Ingot);
    Pair<List<ItemStack>, Integer> gearOre = Pair.of(OreDictionary.getOres("gear" + ore), Material.VALUE_Ingot);

    builder.add(nuggetOre, ingotOre, blockOre, oreOre, plateOre, gearOre);
    Set<Pair<List<ItemStack>, Integer>> knownOres = builder.build();


    // register oredicts
    for(Pair<List<ItemStack>, Integer> pair : knownOres) {
      TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.of(pair.getLeft(), pair.getRight()), fluid));
    }

    // register oredict castings!
    // ingot casting
    TinkerRegistry.registerTableCasting(new OreCastingRecipe(ingotOre.getLeft(),
                                                             RecipeMatch.ofNBT(castIngot),
                                                             fluid,
                                                             ingotOre.getRight()));
    // nugget casting
    TinkerRegistry.registerTableCasting(new OreCastingRecipe(nuggetOre.getLeft(),
                                                             RecipeMatch.ofNBT(castNugget),
                                                             fluid,
                                                             nuggetOre.getRight()));
    // block casting
    TinkerRegistry.registerBasinCasting(new OreCastingRecipe(blockOre.getLeft(),
                                                             null, // no cast
                                                             fluid,
                                                             blockOre.getRight()));
    // plate casting
    TinkerRegistry.registerTableCasting(new OreCastingRecipe(plateOre.getLeft(),
                                                             RecipeMatch.ofNBT(castPlate),
                                                             fluid,
                                                             plateOre.getRight()));
    // gear casting
    TinkerRegistry.registerTableCasting(new OreCastingRecipe(gearOre.getLeft(),
                                                             RecipeMatch.ofNBT(castGear),
                                                             fluid,
                                                             gearOre.getRight()));

    // and also cast creation!
    for(FluidStack fs : castCreationFluids) {
      TinkerRegistry.registerTableCasting(new CastingRecipe(castIngot, RecipeMatch.of(ingotOre.getLeft()), fs, true, true));
      TinkerRegistry.registerTableCasting(new CastingRecipe(castNugget, RecipeMatch.of(nuggetOre.getLeft()), fs, true, true));
      TinkerRegistry.registerTableCasting(new CastingRecipe(castPlate, RecipeMatch.of(plateOre.getLeft()), fs, true, true));
      TinkerRegistry.registerTableCasting(new CastingRecipe(castGear, RecipeMatch.of(gearOre.getLeft()), fs, true, true));
    }

    // used for recipe detection
    knownOreFluids.put(fluid, knownOres);
  }

  // take all fluids we registered oredicts for and scan all recipies for oredict-recipies that we can apply this to
  private static void registerRecipeOredictMelting() {
    // we go through all recipies, and if it's an ore recipe we go through its contents and check if it
    // only consists of one of our known oredict entries
    for(IRecipe irecipe : CraftingManager.func_77594_a().func_77592_b()) {
      // blacklisted?
      boolean blacklisted = false;
      for(ItemStack blacklistItem : meltingBlacklist) {
        if(OreDictionary.itemMatches(blacklistItem, irecipe.func_77571_b(), false)) {
          blacklisted = true;
          break;
        }
      }

      // recipe already has a melting recipe?
      if(blacklisted || TinkerRegistry.getMelting(irecipe.func_77571_b()) != null) {
        continue;
      }

      List<Object> inputs;
      if(irecipe instanceof ShapelessOreRecipe) {
        inputs = ((ShapelessOreRecipe) irecipe).getInput();
      }
      else if(irecipe instanceof ShapedOreRecipe) {
        inputs = Arrays.asList(((ShapedOreRecipe) irecipe).getInput());
      }
      else if(irecipe instanceof ShapelessRecipes) {
        inputs = Lists.<Object>newLinkedList(((ShapelessRecipes) irecipe).field_77579_b);
      }
      else if(irecipe instanceof ShapedRecipes) {
        inputs = Arrays.asList((Object[]) ((ShapedRecipes) irecipe).field_77574_d);
      }
      else {
        // not an ore recipe, stop here because we can't handle it
        continue;
      }

      // this map holds how much of which fluid is known of the recipe
      // if an recipe contains an itemstack that can't be mapped to a fluid calculation is aborted
      Map<Fluid, Integer> known = Maps.newHashMap();
      for(Object o : inputs) {
        // can contain nulls because of shapedrecipe
        if(o == null) {
          continue;
        }
        boolean found = false;
        for(Map.Entry<Fluid, Set<Pair<List<ItemStack>, Integer>>> entry : knownOreFluids.entrySet()) {
          // check if it's a known oredict (all oredict lists are equal if they match the same oredict)
          // OR if it's an itemstack contained in one of our oredicts
          for(Pair<List<ItemStack>, Integer> pair : entry.getValue()) {
            if(o == pair.getLeft() || (o instanceof ItemStack && pair.getLeft().contains(o))) {
              // matches! Update fluid amount known
              Integer amount = known.get(entry.getKey()); // what we found for the liquid so far
              if(amount == null) {
                // nothing is what we found so far.
                amount = 0;
              }
              amount += pair.getRight();
              known.put(entry.getKey(), amount);
              found = true;
              break;
            }
          }
          if(found) {
            break;
          }
        }
        // not a recipe we can process, contains an item that can't melt
        if(!found) {
          known.clear();
          break;
        }
      }

      // add a melting recipe for it
      // we only support single-liquid recipies currently :I
      if(known.keySet().size() == 1) {
        Fluid fluid = known.keySet().iterator().next();
        ItemStack output = irecipe.func_77571_b().func_77946_l();
        int amount = known.get(fluid) / output.field_77994_a;
        output.field_77994_a = 1;
        TinkerRegistry.registerMelting(new MeltingRecipe(RecipeMatch.ofNBT(output, amount), fluid));
        log.trace("Added automatic melting recipe for {} ({} {})", irecipe.func_77571_b().toString(), amount, fluid
            .getName());
      }
    }
  }
}
