package slimeknights.tconstruct.tools.item;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;

import gnu.trove.set.hash.THashSet;

import net.minecraft.block.Block;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemAxe;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.World;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.gameevent.TickEvent;

import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.Stack;

import slimeknights.tconstruct.library.materials.ExtraMaterialStats;
import slimeknights.tconstruct.library.materials.HandleMaterialStats;
import slimeknights.tconstruct.library.materials.HeadMaterialStats;
import slimeknights.tconstruct.library.materials.Material;
import slimeknights.tconstruct.library.tinkering.Category;
import slimeknights.tconstruct.library.tinkering.PartMaterialType;
import slimeknights.tconstruct.library.tools.AoeToolCore;
import slimeknights.tconstruct.library.tools.ToolNBT;
import slimeknights.tconstruct.library.utils.ToolHelper;
import slimeknights.tconstruct.tools.TinkerMaterials;
import slimeknights.tconstruct.tools.TinkerTools;
import slimeknights.tconstruct.tools.events.TinkerToolEvent;
import slimeknights.tconstruct.tools.traits.InfiTool;

public class LumberAxe extends AoeToolCore {

  public static final ImmutableSet<net.minecraft.block.material.Material> effective_materials =
      ImmutableSet.of(net.minecraft.block.material.Material.field_151575_d,
                      net.minecraft.block.material.Material.field_151572_C,
                      net.minecraft.block.material.Material.field_151570_A);

  public LumberAxe() {
    super(PartMaterialType.handle(TinkerTools.toughToolRod),
          PartMaterialType.head(TinkerTools.broadAxeHead),
          PartMaterialType.head(TinkerTools.largePlate),
          PartMaterialType.extra(TinkerTools.toughBinding));

    // lumberaxe is not a weapon. it's for lumberjacks. Lumberjacks are manly, they're weapons themselves.
    addCategory(Category.HARVEST);

    this.setHarvestLevel("axe", 0);
  }

  @Override
  public void func_150895_a(Item itemIn, CreativeTabs tab, List<ItemStack> subItems) {
    addDefaultSubItems(subItems);
    addInfiTool(subItems, "InfiChopper");
  }

  @Override
  public float miningSpeedModifier() {
    return 0.35f; // a bit slower because it breaks whole trees
  }

  @Override
  public float damagePotential() {
    return 0.9f;
  }

  @Override
  public boolean isEffective(Block block) {
    return effective_materials.contains(block.func_149688_o()) || ItemAxe.field_150917_c.contains(block);
  }

  @Override
  public float knockback() {
    return 1.5f;
  }

  @Override
  public boolean onBlockStartBreak(ItemStack itemstack, BlockPos pos, EntityPlayer player) {
    if(ToolHelper.isToolEffective2(itemstack, player.field_70170_p.func_180495_p(pos)) && detectTree(player.field_70170_p, pos)) {
      return fellTree(itemstack, pos, player);
    }
    return super.onBlockStartBreak(itemstack, pos, player);
  }

  @Override
  public ImmutableList<BlockPos> getAOEBlocks(ItemStack stack, World world, EntityPlayer player, BlockPos origin) {
    if(!ToolHelper.isToolEffective2(stack, world.func_180495_p(origin))) {
      return ImmutableList.of();
    }
    return ToolHelper.calcAOEBlocks(stack, world, player, origin, 3, 3, 3);
  }

  @Override
  public int[] getRepairParts() {
    return new int[] {1,2};
  }

  @Override
  public NBTTagCompound buildTag(List<Material> materials) {
    HandleMaterialStats handle = materials.get(0).getStatsOrUnknown(HandleMaterialStats.TYPE);
    HeadMaterialStats head     = materials.get(1).getStatsOrUnknown(HeadMaterialStats.TYPE);
    HeadMaterialStats plate    = materials.get(2).getStatsOrUnknown(HeadMaterialStats.TYPE);
    ExtraMaterialStats binding = materials.get(3).getStatsOrUnknown(ExtraMaterialStats.TYPE);

    ToolNBT data = new ToolNBT();
    data.head(head, plate);
    data.extra(binding);
    data.handle(handle);

    data.durability *= 2f;

    data.modifiers = 2;

    return data.get();
  }

  public static boolean detectTree(World world, BlockPos origin) {
    BlockPos pos = null;
    Stack<BlockPos> candidates = new Stack<BlockPos>();
    candidates.add(origin);

    while(!candidates.isEmpty()) {
      BlockPos candidate = candidates.pop();
      if((pos == null || candidate.func_177956_o() > pos.func_177956_o()) && isLog(world, candidate)) {
        pos = candidate.func_177984_a();
        // go up
        while(isLog(world, pos)) {
          pos = pos.func_177984_a();
        }
        // check if we still have a way diagonally up
        candidates.add(pos.func_177978_c());
        candidates.add(pos.func_177974_f());
        candidates.add(pos.func_177968_d());
        candidates.add(pos.func_177976_e());
      }
    }

    // not even one match, so there were no logs.
    if(pos == null) {
      return false;
    }

    // check if there were enough leaves around the last position
    // pos now contains the block above the topmost log
    // we want at least 5 leaves in the surrounding 26 blocks
    int d = 3;
    int o = -1; // -(d-1)/2
    int leaves = 0;
    for(int x = 0; x < d; x++) {
      for(int y = 0; y < d; y++) {
        for(int z = 0; z < d; z++) {
          BlockPos leaf = pos.func_177982_a(o + x, o + y, o + z);
          if(world.func_180495_p(leaf).func_177230_c().isLeaves(world, leaf)) {
            if(++leaves >= 5) {
              return true;
            }
          }
        }
      }
    }

    // not enough leaves. sorreh
    return false;
  }

  private static boolean isLog(World world, BlockPos pos) {
    return world.func_180495_p(pos).func_177230_c().isWood(world, pos);
  }

  public static boolean fellTree(ItemStack itemstack, BlockPos start, EntityPlayer player) {
    if(player.field_70170_p.field_72995_K) {
      return true;
    }
    TinkerToolEvent.ExtraBlockBreak event = TinkerToolEvent.ExtraBlockBreak.fireEvent(itemstack, player, 3, 3, 3, -1);
    int speed = Math.round((event.width * event.height * event.depth)/27f);
    if(event.distance > 0) {
      speed = event.distance + 1;
    }

    MinecraftForge.EVENT_BUS.register(new TreeChopTask(itemstack, start, player, speed));
    return true;
  }

  public static class TreeChopTask {

    public final World world;
    public final EntityPlayer player;
    public final ItemStack tool;
    public final int blocksPerTick;

    public Queue<BlockPos> blocks = Lists.newLinkedList();
    public Set<BlockPos> visited = new THashSet<BlockPos>();


    public TreeChopTask(ItemStack tool, BlockPos start, EntityPlayer player, int blocksPerTick) {
      this.world = player.func_130014_f_();
      this.player = player;
      this.tool = tool;
      this.blocksPerTick = blocksPerTick;

      this.blocks.add(start);
    }

    @SubscribeEvent
    public void chopChop(TickEvent.WorldTickEvent event) {
      if(event.side.isClient()) {
        finish();
        return;
      }

      // setup
      int left = blocksPerTick;

      // continue running
      BlockPos pos;
      while(left > 0) {
        // completely done or can't do our job anymore?!
        if(blocks.isEmpty() || ToolHelper.isBroken(tool)) {
          finish();
          return;
        }

        pos = blocks.remove();
        if(!visited.add(pos)) {
          continue;
        }

        // can we harvest the block and is effective?
        if(!isLog(world, pos) || !ToolHelper.isToolEffective2(tool, world.func_180495_p(pos))) {
          continue;
        }

        // save its neighbours
        for(EnumFacing facing : new EnumFacing[]{EnumFacing.NORTH, EnumFacing.EAST, EnumFacing.SOUTH, EnumFacing.WEST}) {
          BlockPos pos2 = pos.func_177972_a(facing);
          if(!visited.contains(pos2)) {
            blocks.add(pos2);
          }
        }

        // also add the layer above.. stupid acacia trees
        for(int x = 0; x < 3; x++) {
          for(int z = 0; z < 3; z++) {
            BlockPos pos2 = pos.func_177982_a(-1 + x, 1, -1 + z);
            if(!visited.contains(pos2)) {
              blocks.add(pos2);
            }
          }
        }

        // break it, wooo!
        ToolHelper.breakExtraBlock(tool, world, player, pos, pos);
        left--;
      }
    }

    private void finish() {
      // goodbye cruel world
      MinecraftForge.EVENT_BUS.unregister(this);
    }
  }
}
