package slimeknights.tconstruct.library.modifiers;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumChatFormatting;

import slimeknights.tconstruct.library.TinkerRegistry;
import slimeknights.tconstruct.library.Util;
import slimeknights.tconstruct.library.client.CustomFontColor;

/**
 * Represents the NBT data saved for a modifier.
 */
public class ModifierNBT {

  public String identifier;
  public int color;
  public int level;
  public String extraInfo;

  public ModifierNBT() {
    identifier = "";
    color = 0xffffff;
    level = 0;
  }

  public ModifierNBT(IModifier modifier) {
    this.identifier = modifier.getIdentifier();
    this.level = 0;
    this.color = Util.enumChatFormattingToColor(EnumChatFormatting.GRAY);
  }

  public ModifierNBT(NBTTagCompound tag) {
    this();
    read(tag);
  }

  public static ModifierNBT readTag(NBTTagCompound tag) {
    ModifierNBT data = new ModifierNBT();
    if(tag != null) {
      data.read(tag);
    }

    return data;
  }

  public void read(NBTTagCompound tag) {
    identifier = tag.func_74779_i("identifier");
    color = tag.func_74762_e("color");
    level = tag.func_74762_e("level");
    extraInfo = tag.func_74779_i("extraInfo");
  }

  public void write(NBTTagCompound tag) {
    tag.func_74778_a("identifier", identifier);
    tag.func_74768_a("color", color);
    if(level > 0) {
      tag.func_74768_a("level", level);
    }
    else {
      tag.func_82580_o("level");
    }
    if(extraInfo != null && !extraInfo.isEmpty()) {
      tag.func_74778_a("extraInfo", extraInfo);
    }
  }

  public String getColorString() {
    return CustomFontColor.encodeColor(color);
  }

  public static <T extends ModifierNBT> T readTag(NBTTagCompound tag, Class<T> clazz) {
    try {
      T data = clazz.newInstance();
      data.read(tag);
      return data;
    } catch(ReflectiveOperationException e) {
      TinkerRegistry.log.error(e);
      return null;
    }
  }

  public static IntegerNBT readInteger(NBTTagCompound tag) {
    return ModifierNBT.readTag(tag, IntegerNBT.class);
  }

  public static BooleanNBT readBoolean(NBTTagCompound tag) {
    return ModifierNBT.readTag(tag, BooleanNBT.class);
  }


  /**
   * Single boolean value
   */
  public static class BooleanNBT extends ModifierNBT {

    public boolean status;

    public BooleanNBT() {
    }

    public BooleanNBT(IModifier modifier, boolean status) {
      super(modifier);
      this.status = status;
    }

    @Override
    public void write(NBTTagCompound tag) {
      super.write(tag);
      tag.func_74757_a("status", status);
    }

    @Override
    public void read(NBTTagCompound tag) {
      super.read(tag);
      status = tag.func_74767_n("status");
    }
  }

  /**
   * Data can be applied multiple times up to a maximum.
   */
  public static class IntegerNBT extends ModifierNBT {

    public int current;
    public int max;

    public IntegerNBT() {
    }

    public IntegerNBT(IModifier modifier, int current, int max) {
      super(modifier);
      this.current = current;
      this.max = max;

      this.extraInfo = calcInfo();
    }

    @Override
    public void write(NBTTagCompound tag) {
      calcInfo();
      super.write(tag);
      tag.func_74768_a("current", current);
      tag.func_74768_a("max", max);
    }

    @Override
    public void read(NBTTagCompound tag) {
      super.read(tag);
      current = tag.func_74762_e("current");
      max = tag.func_74762_e("max");

      extraInfo = calcInfo();
    }

    public String calcInfo() {
      if(max > 0) {
        return String.format("%d / %d", current, max);
      }

      return current > 0 ? String.valueOf(current) : "";
    }
  }
}
