package slimeknights.tconstruct.smeltery.tileentity;

import net.minecraft.block.state.IBlockState;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.Packet;
import net.minecraft.network.play.server.S35PacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ITickable;
import net.minecraft.world.WorldServer;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.IFluidHandler;

import slimeknights.tconstruct.common.TinkerNetwork;
import slimeknights.tconstruct.library.materials.Material;
import slimeknights.tconstruct.smeltery.block.BlockFaucet;
import slimeknights.tconstruct.smeltery.network.FaucetActivationPacket;

public class TileFaucet extends TileEntity implements ITickable {

  public static final int LIQUID_TRANSFER = 6; // how much liquid is transferred per operation
  public static final int TRANSACTION_AMOUNT = Material.VALUE_Ingot;

  // direction we're pulling liquid from. cached so we don't have to query the world every time. set on pour-begin
  public EnumFacing direction; // used to continue draining and for rendering
  public boolean isPouring;
  public boolean stopPouring;
  public FluidStack drained; // fluid is drained instantly and distributed over time. how much is left
  public boolean lastRedstoneState;

  public TileFaucet() {
    reset();
  }

  // begin pouring
  public boolean activate() {
    IBlockState state = field_145850_b.func_180495_p(field_174879_c);
    // invalid state
    if(!state.func_177227_a().contains(BlockFaucet.FACING)) {
      return false;
    }

    // already pouring? we want to stop then
    if(isPouring) {
      stopPouring = true;
      return true;
    }

    direction = field_145850_b.func_180495_p(field_174879_c).func_177229_b(BlockFaucet.FACING);
    doTransfer();
    return isPouring;
  }

  public void handleRedstone(boolean hasSignal) {
    if(hasSignal != lastRedstoneState) {
      lastRedstoneState = hasSignal;
      if(hasSignal) {
        activate();
      }
    }
  }

  @Override
  public void func_73660_a() {
    if(field_145850_b.field_72995_K) {
      return;
    }
    // nothing to do if not pouring
    if(!isPouring) {
      return;
    }

    if(drained != null) {
      // done draining
      if(drained.amount <= 0) {
        drained = null;
        // pour me another, if we want to.
        if(!stopPouring) {
          doTransfer();
        }
        else {
          reset();
        }
      }
      else {
        // reduce amount (cooldown)
        pour();
      }
    }
  }

  protected void doTransfer() {
    // still got content left
    if(drained != null) {
      return;
    }
    TileEntity drainTE = field_145850_b.func_175625_s(field_174879_c.func_177972_a(direction));
    TileEntity fillTE = field_145850_b.func_175625_s(field_174879_c.func_177977_b());
    if(drainTE instanceof IFluidHandler && fillTE instanceof IFluidHandler) {
      IFluidHandler toDrain = (IFluidHandler) drainTE;
      IFluidHandler toFill = (IFluidHandler) fillTE;

      // can we drain?
      FluidStack drained = toDrain.drain(direction, TRANSACTION_AMOUNT, false);
      if(drained != null) {
        // can we fill?
        int filled = toFill.fill(EnumFacing.UP, drained, false);
        if(filled > 0) {
          // drain the liquid and transfer it, buffer the amount for delay
          this.drained = toDrain.drain(direction, filled, true);
          this.isPouring = true;
          pour();

          // sync to clients
          if(!field_145850_b.field_72995_K && field_145850_b instanceof WorldServer) {
            TinkerNetwork.sendToClients((WorldServer) field_145850_b, field_174879_c, new FaucetActivationPacket(field_174879_c, drained));
          }

          return;
        }
      }
    }
    // draining unsuccessful
    reset();
  }

  // takes the liquid inside and executes one pouring step
  protected void pour() {
    if(drained == null) {
      return;
    }

    TileEntity fillTE = field_145850_b.func_175625_s(field_174879_c.func_177977_b());
    if(fillTE instanceof IFluidHandler) {
      IFluidHandler toFill = (IFluidHandler) fillTE;

      FluidStack fillStack = drained.copy();
      fillStack.amount = Math.min(drained.amount, LIQUID_TRANSFER);

      // can we fill?
      int filled = toFill.fill(EnumFacing.UP, fillStack, false);
      if(filled > 0) {
        // transfer it
        this.drained.amount -= filled;
        fillStack.amount = filled;
        toFill.fill(EnumFacing.UP, fillStack, true);
      }
    }
    else {
      // filling TE got lost. reset. all liquid buffered is lost.
      reset();
    }
  }

  protected void reset() {
    isPouring = false;
    stopPouring = false;
    drained = null;
    direction = EnumFacing.DOWN; // invalid direction
    lastRedstoneState = false;

    // sync to clients
    if(field_145850_b != null && !field_145850_b.field_72995_K && field_145850_b instanceof WorldServer) {
      TinkerNetwork.sendToClients((WorldServer) field_145850_b, field_174879_c, new FaucetActivationPacket(field_174879_c, null));
    }
  }


  /* Load & Save */

  @Override
  public void func_145841_b(NBTTagCompound compound) {
    super.func_145841_b(compound);
    if(drained != null) {
      drained.writeToNBT(compound);
      compound.func_74768_a("direction", direction.func_176745_a());
      //compound.setString("direction", direction.getName());
      compound.func_74757_a("stop", stopPouring);
    }
  }

  @Override
  public void func_145839_a(NBTTagCompound compound) {
    super.func_145839_a(compound);
    drained = FluidStack.loadFluidStackFromNBT(compound);

    if(drained != null) {
      isPouring = true;
      direction = EnumFacing.values()[compound.func_74762_e("direction")];
      //direction = EnumFacing.valueOf(compound.getString("direction"));
      stopPouring = compound.func_74767_n("stop");
    }
    else {
      reset();
    }
  }

  public void onActivationPacket(FluidStack fluid) {
    if(fluid == null) {
      reset();
    }
    else {
      drained = fluid;
      isPouring = true;
      direction = field_145850_b.func_180495_p(field_174879_c).func_177229_b(BlockFaucet.FACING);
    }
  }

  @Override
  public Packet func_145844_m() {
    NBTTagCompound tag = new NBTTagCompound();
    func_145841_b(tag);
    return new S35PacketUpdateTileEntity(this.func_174877_v(), this.func_145832_p(), tag);
  }

  @Override
  public void onDataPacket(NetworkManager net, S35PacketUpdateTileEntity pkt) {
    super.onDataPacket(net, pkt);
    func_145839_a(pkt.func_148857_g());
  }
}
